// Copyright 2014 Google Inc. All Rights Reserved.

#include "Controller.h"

int Controller::routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg) {
    int ret = STATUS_UNEXPECTED_MESSAGE;
    uint8_t* ptr = (uint8_t*)msg->raw() + sizeof(uint16_t);
    size_t len = msg->size() - sizeof(uint16_t);

    switch (type) {
        case MESSAGE_VERSION_RESPONSE:
            ret = handleVersionResponse(ptr, len);
            if (mHealthy) {
                startAuthentication();
            }
            break;
        case MESSAGE_SERVICE_DISCOVERY_REQUEST: {
            ServiceDiscoveryRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleServiceDiscoveryRequest(req);
            }
            break;
        }
        case MESSAGE_PING_REQUEST: {
            PingRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handlePingRequest(req);
            }
            break;
        }
        case MESSAGE_PING_RESPONSE: {
            PingResponse resp;
            if (PARSE_PROTO(resp, ptr, len)) {
                ret = handlePingResponse(resp);
            }
            break;
        }
        case MESSAGE_ENCAPSULATED_SSL: {
            IoBuffer out;
            ret = mSslWrapper.handshake(ptr, len, &out);
            if (ret < 0 && out.size() > 0) {
                queueOutgoingUnencrypted(out.raw(), out.size());
            } else if (ret == 1 && mSslWrapper.verifyPeer()) {
                mRouter->setSslWrapper(&mSslWrapper);
                sendAuthResponse(STATUS_SUCCESS);
            } else {
                sendAuthResponse(STATUS_AUTHENTICATION_FAILURE);
                LOGE("Shutting down connection due to authentication failure.");
                unrecoverableError(STATUS_AUTHENTICATION_FAILURE);
            }
            ret = STATUS_SUCCESS;
            break;
        }
        case MESSAGE_NAV_FOCUS_REQUEST: {
            NavFocusRequestNotification req;
            if (PARSE_PROTO(req, ptr, len)) {
                mControllerCallbacks->navigationFocusCallback(req.focus_type());
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case MESSAGE_BYEBYE_REQUEST: {
            ByeByeRequest request;
            if (PARSE_PROTO(request, ptr, len)) {
                mControllerCallbacks->byeByeRequestCallback(request.reason());
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case MESSAGE_BYEBYE_RESPONSE: {
            ByeByeResponse response;
            if (PARSE_PROTO(response, ptr, len)) {
                mControllerCallbacks->byeByeResponseCallback();
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case MESSAGE_VOICE_SESSION_NOTIFICATION: {
            VoiceSessionNotification notif;
            if (PARSE_PROTO(notif, ptr, len)) {
                mControllerCallbacks->voiceSessionNotificationCallback(notif.status());
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case MESSAGE_AUDIO_FOCUS_REQUEST: {
            AudioFocusRequestNotification req;
            if (PARSE_PROTO(req, ptr, len)) {
                mControllerCallbacks->audioFocusRequestCallback(req.request());
                ret = STATUS_SUCCESS;
            }
            break;
        }
    }
    return ret;
}

void Controller::sendAuthResponse(int status) {
    AuthResponse resp;
    resp.set_status(status);
    IoBuffer buf;
    mRouter->marshallProto(MESSAGE_AUTH_COMPLETE, resp, &buf);
    queueOutgoingUnencrypted(buf.raw(), buf.size());
}

void Controller::start() {
    // The head unit starts the protocol when it detects the usb connection.
    sendVersionRequest();
}

int Controller::sendVersionRequest() {
    size_t len = 3 * sizeof(uint16_t);
    uint8_t* buf = new uint8_t[len];
    WRITE_BE16(buf, MESSAGE_VERSION_REQUEST);
    WRITE_BE16(buf + 2, PROTOCOL_MAJOR_VERSION);
    WRITE_BE16(buf + 4, PROTOCOL_MINOR_VERSION);
    queueOutgoingUnencrypted(buf, len);
    delete[] buf;
    LOG("Starting link, car major=%hu minor=%hu", PROTOCOL_MAJOR_VERSION, PROTOCOL_MINOR_VERSION);
    return STATUS_SUCCESS;
}

int Controller::handleVersionResponse(void* msg, size_t len) {
    uint8_t *ptr = (uint8_t *) msg;
    uint16_t major;
    READ_BE16(ptr, major);
    uint16_t minor;
    READ_BE16(ptr + 2, minor);
    uint8_t status = *(ptr + 4);

    LOG("Phone reported version major=%hu minor=%hu", major, minor);
    mHealthy = (status == STATUS_SUCCESS);
    return STATUS_SUCCESS;
}

int Controller::startAuthentication() {
    IoBuffer buf;
    if (mSslWrapper.handshake(NULL, 0, &buf) < 0) {
        queueOutgoingUnencrypted(buf.raw(), buf.size());
        return STATUS_SUCCESS;
    }
    return STATUS_INTERNAL_ERROR;
}

int Controller::handleServiceDiscoveryRequest(const ServiceDiscoveryRequest& req) {
    if (mHealthy == false) {
        return STATUS_UNEXPECTED_MESSAGE;
    }
    mControllerCallbacks->serviceDiscoveryRequestCallback(req.small_icon(), req.medium_icon(),
            req.large_icon(), req.label_text(), req.device_name());
    sendServiceDiscoveryResponse();
    return STATUS_SUCCESS;
}

int Controller::sendServiceDiscoveryResponse() {
    ServiceDiscoveryResponse sdr;
    sdr.set_make(mMake);
    sdr.set_model(mModel);
    sdr.set_year(mYear);
    sdr.set_vehicle_id(mId);
    sdr.set_driver_position((DriverPosition) mDriverPosition);
    sdr.set_head_unit_make(mHuMake);
    sdr.set_head_unit_model(mHuModel);
    sdr.set_head_unit_software_build(mHuSwBuild);
    sdr.set_head_unit_software_version(mHuSwVersion);
    sdr.set_session_configuration(mSessionConfiguration);
    mRouter->populateServiceDiscoveryResponse(&sdr);
    IoBuffer buf;
    mRouter->marshallProto(MESSAGE_SERVICE_DISCOVERY_RESPONSE, sdr, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int Controller::handlePingRequest(const PingRequest& req) {
    bool bugReport = req.has_bug_report() && req.bug_report();
    mControllerCallbacks->pingRequestCallback(req.timestamp(), bugReport);
    PingResponse resp;
    resp.set_timestamp(req.timestamp());
    IoBuffer buf;
    mRouter->marshallProto(MESSAGE_PING_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int Controller::handlePingResponse(const PingResponse& resp) {
    mControllerCallbacks->pingResponseCallback(resp.timestamp());
    return STATUS_SUCCESS;
}

void Controller::sendPingRequest(int64_t timestamp, bool bugReport) {
    PingRequest req;
    req.set_timestamp(timestamp);
    req.set_bug_report(bugReport);
    IoBuffer buf;
    mRouter->marshallProto(MESSAGE_PING_REQUEST, req, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

bool Controller::setClientCreds(const string& rootCert, const string& clientCert,
        const string& privKey) {
    return mSslWrapper.init(rootCert, clientCert, privKey);
}

void Controller::setNavigationFocus(NavFocusType type) {
    NavFocusNotification notification;
    notification.set_focus_type(type);
    IoBuffer buf;
    mRouter->marshallProto(MESSAGE_NAV_FOCUS_NOTIFICATION, notification, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

void Controller::sendByeByeRequest(ByeByeReason reason) {
    ByeByeRequest request;
    request.set_reason(reason);
    IoBuffer buf;
    mRouter->marshallProto(MESSAGE_BYEBYE_REQUEST, request, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

void Controller::sendByeByeResponse() {
    ByeByeResponse response;
    IoBuffer buf;
    mRouter->marshallProto(MESSAGE_BYEBYE_RESPONSE, response, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

void Controller::setAudioFocus(AudioFocusStateType focusState, bool unsolicited) {
    AudioFocusNotification afn;
    afn.set_focus_state(focusState);
    afn.set_unsolicited(unsolicited);
    IoBuffer buf;
    mRouter->marshallProto(MESSAGE_AUDIO_FOCUS_NOTIFICATION, afn, &buf);
    queueOutgoing(buf.raw(), buf.size());
}
